#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
import logging
from multiprocessing import Manager, Process, Value
import os
from pathlib import Path
import sys
import threading
import time

from core.programconfig import ProgramConfig
import serial
from version import GetVersion
from work import fileSendLoop, scanAndSelect


def endProgram(pause, code):
    if pause != '0':
        logging.info('Press enter to exit program')
        input()
    else:
        logging.info('Exit program')
    sys.exit(code)

# start
if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='version: '+GetVersion())
    parser.add_argument("-r", "--retry_max", type=int,
                        help="maximum retry count, default 50", default=50)
    parser.add_argument("-v", "--verbose", help="increase output verbosity",
                        action="store_true")
    parser.add_argument("-c", "--config", metavar='usb_booting.txt',
                        help="configuration file name", nargs='?',
                        default='usb_booting.txt')

    args = parser.parse_args()

    if args.verbose:
        logging.basicConfig(level=logging.DEBUG)
    else:
        logging.basicConfig(level=logging.INFO)

    configFile = args.config
    cfg_file = Path(os.path.realpath(configFile))
    if not cfg_file.is_file():
        parser.print_help()
        sys.exit(0)

    ser = serial.Serial()


    cfg = ProgramConfig(configFile)
    cfg.parse()

    if cfg.option_program_loop == '0':
        looping = False
    else:
        looping = True

    workThreadList = []
    while True:
        selectedPort, opened_ports = scanAndSelect(cfg)

        if cfg.option_force_device is not None:
            fileSendLoop(selectedPort, cfg.fileList, cfg.option_stop_program_on_fail_response, args.retry_max)
        elif cfg.option_auto_start == '0':
            fileSendLoop(selectedPort, cfg.fileList, cfg.option_stop_program_on_fail_response, args.retry_max)
        elif cfg.option_auto_start == '1':
            for selectedPort in opened_ports:
                logging.debug('*Start work on '+selectedPort)
                #fileSendLoop(selectedPort)

                #using thread
                workThread = threading.Thread(target=fileSendLoop,
                                              args=(selectedPort, cfg.fileList, cfg.option_stop_program_on_fail_response, args.retry_max))
                workThread.daemon = True
                workThread.start()
                workThreadList.append(workThread)

                """
                # using multiprocess <- cannot use for one binary with Pyinstaller
                # Warning The 'spawn' and 'forkserver' start methods cannot currently be used with “frozen” executables
                # (i.e., binaries produced by packages like PyInstaller and cx_Freeze) on Unix. The 'fork' start method does work.
                # spawn <- windows default

                manager = Manager()
                managed_fileList = manager.dict(cfg.fileList)
                p = Process(target=fileSendLoop, args=(selectedPort, managed_fileList), daemon=True,
                            name=('process_'+selectedPort))
                p.start()
                p.join()
                workThreadList.append(p)
                """

        #  wait until all threads terminated.
        isAliveAll = 1
        while isAliveAll != 0:
            time.sleep(1)
            isAliveAll = 0
            for aThread in workThreadList:
                if aThread.is_alive() == True:
                    isAliveAll += 1
        logging.info('All work threads finished.')

        if looping:
            continue

        break

    endProgram(cfg.option_pause_before_terminate, 0)
