#ifndef _CRWQUEUE_H_
#define _CRWQUEUE_H_


/////////////////////////////////////////////////////////////////////////////


#include <new>
#include <Windows.h>
#include <Limits.h>


/////////////////////////////////////////////////////////////////////////////


/// Сдп
#define MIN_RW_QUEUE_BLOCK_CAPACITY                         (1024 * 1)

/// Ĭ϶дп
#define DEFAULT_RW_QUEUE_BLOCK_CAPACITY                     (1024 * 64)

#ifdef WIN64                    /// 64Bit

/// дп
#define MAX_RW_QUEUE_BLOCK_CAPACITY                         (((ULONGLONG)-1) / 4)

#else                           /// 32Bit

/// дп
#define MAX_RW_QUEUE_BLOCK_CAPACITY                         (((UINT)-1) / 4)

#endif


/////////////////////////////////////////////////////////////////////////////


/// дУ̰߳ȫһ̶߳Сһ߳дУߡ⣩
/// @param TItem - Ԫ
template<typename TItem>
class CRWQueue
{
public:

    /// 캯̰߳ȫʱ
    /// @param uBlockCapacity - пMIN_RW_QUEUE_BLOCK_CAPACITY  MAX_RW_QUEUE_BLOCK_CAPACITYĬֵ=DEFAULT_RW_QUEUE_BLOCK_CAPACITY
    /// @param bSetAllBlockItemDataZero - TRUE=ȫöпԪΪ 0FALSE=ȫöпԪΪ 0Ĭֵ
    /// @param bMemoryCopyItem - TRUE=ڴ濽ԪأĬֵFALSE=ڴ濽Ԫ
    CRWQueue(size_t uBlockCapacity = DEFAULT_RW_QUEUE_BLOCK_CAPACITY,
        BOOL bSetAllBlockItemDataZero = FALSE, BOOL bMemoryCopyItem = TRUE);

    /// ̰߳ȫʱ
    virtual ~CRWQueue(VOID);

    /// ȡöп
    /// @return п
    size_t BlockCapacity(VOID) CONST;

    /// ȡöдС
    /// @return дС
    size_t Size(VOID) CONST;

    /// Ƿǿն
    /// @return TRUE=նУFALSE=ǿն
    BOOL IsEmpty(VOID) CONST;

    /// ȡöпʼԪ
    /// @return пʼԪ
    TItem *Begin(VOID);

    /// ȡöнԪ
    /// @return нԪ
    TItem *End(VOID);

    /// β
    /// @param pItems - Ԫؼ NULL
    /// @param uSize - ӴСпĬֵ=1
    /// @return ʵӴС
    size_t Push(CONST TItem *pItems, size_t uSize = 1);

    /// β
    /// @param tItem - ԪأԪأ
    /// @return ʵӴС
    size_t Push(CONST TItem &tItem);

    /// ׳
    /// @param pItems - NULL=Ч NULL=Ԫؼ
    /// @param uSize - ӴС pItems СпĬֵ=SIZE_MAX
    /// @return ʵʳӴС
    size_t Pop(TItem *pItems, size_t uSize = SIZE_MAX);

    /// ׳
    /// @param tItem - ԪأԪأ
    /// @return TRUE=ɹFALSE=ʧ
    BOOL Pop(TItem &tItem);

    /// գ̰߳ȫʱ
    VOID Clear(VOID);

private:

    /// 
    typedef struct TBlockQueue
    {
    public:

        /// 
        TItem *pQueue;

        /// һ
        TBlockQueue *pNext;


        /// 캯
        /// @param uBlockCapacity - пMIN_RW_QUEUE_BLOCK_CAPACITY  MAX_RW_QUEUE_BLOCK_CAPACITYĬֵ=DEFAULT_RW_QUEUE_BLOCK_CAPACITY
        /// @param bSetAllBlockItemDataZero - TRUE=ȫöпԪΪ 0FALSE=ȫöпԪΪ 0Ĭֵ
        TBlockQueue(size_t uBlockCapacity, BOOL bSetAllBlockItemDataZero)
            : pNext(NULL)
        {
#ifdef _DEBUG

            pQueue = new TItem[uBlockCapacity];

#else

            pQueue = new(std::nothrow) TItem[uBlockCapacity];

#endif

            if (bSetAllBlockItemDataZero && pQueue)
            {
                ::ZeroMemory(pQueue, sizeof(TItem) * uBlockCapacity);
            }
        }


        /// 
        ~TBlockQueue(VOID)
        {
            pNext = NULL;

            if (pQueue)
            {
                delete []pQueue;
                pQueue = NULL;
            }
        }
    }*TBlockQueuePtr;


    /// ͷ
    volatile TBlockQueuePtr m_pHeadBlockQueue;

    /// β
    volatile TBlockQueuePtr m_pTailBlockQueue;

    /// п
    CONST size_t m_uBlockCapacity;

    /// ǷȫöпԪΪ 0TRUE=ȫöпԪΪ 0FALSE=ȫöпԪΪ 0Ĭֵ
    CONST BOOL m_bSetAllBlockItemDataZero;

    /// Ƿڴ濽ԪأTRUE=ڴ濽ԪأĬֵFALSE=ڴ濽Ԫ
    CONST BOOL m_bMemoryCopyItem;

    /// жָ
    volatile UINT64 m_uReadPointer;

    /// дָ
    volatile UINT64 m_uWritePointer;


    /// 캯ⲿ
    /// @param other - һдж
    CRWQueue(CONST CRWQueue &other);

    /// ֵⲿֵ
    /// @param other - һдж
    /// @return дж
    CRWQueue &operator=(CONST CRWQueue &other);


    /// ͷ
    VOID CheckHeadBlockQueue(VOID);

    /// Ԫ
    /// @param pSourceItems - ԴԪؼ NULL
    /// @param pTargetItems - ĿԪؼ NULL
    /// @param uSize - С
    VOID CopyItem(CONST TItem *pSourceItems, TItem *pTargetItems, size_t uSize);
};


template<typename TItem>
CRWQueue<TItem>::CRWQueue(size_t uBlockCapacity, BOOL bSetAllBlockItemDataZero, BOOL bMemoryCopyItem)
    : m_uBlockCapacity(max(min(uBlockCapacity, MAX_RW_QUEUE_BLOCK_CAPACITY), MIN_RW_QUEUE_BLOCK_CAPACITY))
    , m_bSetAllBlockItemDataZero(bSetAllBlockItemDataZero)
    , m_bMemoryCopyItem(bMemoryCopyItem)
    , m_uReadPointer(0)
    , m_uWritePointer(0)
{
#ifdef _DEBUG

    m_pHeadBlockQueue = new TBlockQueue(m_uBlockCapacity, m_bSetAllBlockItemDataZero);

#else

    m_pHeadBlockQueue = new(std::nothrow) TBlockQueue(m_uBlockCapacity, m_bSetAllBlockItemDataZero);

#endif

    m_pTailBlockQueue = m_pHeadBlockQueue;
}


template<typename TItem>
CRWQueue<TItem>::~CRWQueue(VOID)
{
    Clear();
    m_pTailBlockQueue = NULL;

    if (m_pHeadBlockQueue)
    {
        delete m_pHeadBlockQueue;
        m_pHeadBlockQueue = NULL;
    }
}


template<typename TItem>
inline size_t CRWQueue<TItem>::BlockCapacity(VOID) CONST
{
    return m_uBlockCapacity;
}


template<typename TItem>
inline size_t CRWQueue<TItem>::Size(VOID) CONST
{
    return (size_t)(m_uWritePointer - m_uReadPointer);
}


template<typename TItem>
inline BOOL CRWQueue<TItem>::IsEmpty(VOID) CONST
{
    return (m_uWritePointer <= m_uReadPointer);
}


template<typename TItem>
inline TItem *CRWQueue<TItem>::Begin(VOID)
{
    if (!m_pHeadBlockQueue || !m_pHeadBlockQueue->pQueue || IsEmpty())
    {
        return NULL;
    }

    /// ͷٽ硢Խ
    CheckHeadBlockQueue();

    return (m_pHeadBlockQueue->pQueue + m_uReadPointer % m_uBlockCapacity);
}


template<typename TItem>
inline TItem *CRWQueue<TItem>::End(VOID)
{
    if (!m_pTailBlockQueue || !m_pTailBlockQueue->pQueue || IsEmpty())
    {
        return NULL;
    }

    return (m_pTailBlockQueue->pQueue + (m_uWritePointer - 1) % m_uBlockCapacity);
}


template<typename TItem>
VOID CRWQueue<TItem>::CheckHeadBlockQueue(VOID)
{
    /// 
    size_t uIndex = m_uReadPointer % m_uBlockCapacity;

    /// ͷſͷԴ
    if (m_pHeadBlockQueue
        && (m_uReadPointer > 0)
        && (uIndex == 0))
    {
        TBlockQueuePtr pTemp = m_pHeadBlockQueue;
        m_pHeadBlockQueue = m_pHeadBlockQueue->pNext;
        delete pTemp;
        pTemp = NULL;
    }
}


template<typename TItem>
VOID CRWQueue<TItem>::CopyItem(CONST TItem *pSourceItems, TItem *pTargetItems, size_t uSize)
{
    if (pSourceItems && pTargetItems && (pSourceItems != pTargetItems))
    {
        if (m_bMemoryCopyItem)
        {
            ::memcpy(pTargetItems, pSourceItems, sizeof(TItem) * uSize);
        }
        else
        {
            for (size_t i = 0; i < uSize; i++)
            {
                *(pTargetItems + i) = *(pSourceItems + i);
            }
        }
    }
}


template<typename TItem>
size_t CRWQueue<TItem>::Push(CONST TItem *pItems, size_t uSize)
{
    if (!m_pTailBlockQueue || !m_pTailBlockQueue->pQueue || !pItems || (uSize == 0))
    {
        return 0;
    }

    /// 
    size_t uReturned = min(uSize, m_uBlockCapacity);
    size_t uIndex = m_uWritePointer % m_uBlockCapacity;
    uSize = uIndex + uReturned;

    /// δԽ
    if (((uIndex > 0) || (m_uWritePointer < 1)) && (uSize <= m_uBlockCapacity))
    {
        CopyItem(pItems, m_pTailBlockQueue->pQueue + uIndex, uReturned);
    }

    /// Խ硢
    else
    {
        size_t uNewIndex = uSize % m_uBlockCapacity;
        uSize = uReturned - uNewIndex;
        CopyItem(pItems, m_pTailBlockQueue->pQueue + uIndex, uSize);
        m_pTailBlockQueue->pNext = new TBlockQueue(m_uBlockCapacity, m_bSetAllBlockItemDataZero);
        m_pTailBlockQueue = m_pTailBlockQueue->pNext;
        CopyItem(pItems + uSize, m_pTailBlockQueue->pQueue, uNewIndex);
    }

    m_uWritePointer += uReturned;

    return uReturned;
}


template<typename TItem>
inline size_t CRWQueue<TItem>::Push(CONST TItem &tItem)
{
    return Push(&tItem, 1);
}


template<typename TItem>
size_t CRWQueue<TItem>::Pop(TItem *pItems, size_t uSize)
{
    if (!m_pHeadBlockQueue || !m_pHeadBlockQueue->pQueue || IsEmpty() || !pItems || (uSize == 0))
    {
        return 0;
    }

    /// ͷٽ硢Խ
    CheckHeadBlockQueue();

    /// С
    size_t uReturned = Size();
    uSize = min(uSize, m_uBlockCapacity);
    uReturned = min(uSize, uReturned);

    /// 
    size_t uIndex = m_uReadPointer % m_uBlockCapacity;

    /// Խ硢ض
    if ((uReturned + uIndex) > m_uBlockCapacity)
    {
        uReturned = m_uBlockCapacity - uIndex;
    }

    /// 
    CopyItem(m_pHeadBlockQueue->pQueue + uIndex, pItems, uReturned);

    /// Ӷָ
    m_uReadPointer += uReturned;

    return uReturned;
}


template<typename TItem>
inline BOOL CRWQueue<TItem>::Pop(TItem &tItem)
{
    return (Pop(&tItem, 1) > 0);
}


template<typename TItem>
VOID CRWQueue<TItem>::Clear(VOID)
{
    m_uWritePointer = 0;
    m_uReadPointer = 0;

    if (m_pHeadBlockQueue)
    {
        /// ضϿ
        TBlockQueuePtr pHeadBlockQueue = m_pHeadBlockQueue->pNext;
        m_pTailBlockQueue = m_pHeadBlockQueue;
        m_pHeadBlockQueue->pNext = NULL;

        /// ͷſԴ
        while (pHeadBlockQueue)
        {
            TBlockQueuePtr pTemp = pHeadBlockQueue;
            pHeadBlockQueue = pHeadBlockQueue->pNext;
            delete pTemp;
            pTemp = NULL;
        }
    }
}


/////////////////////////////////////////////////////////////////////////////


/// ASCII ַд
typedef CRWQueue<CHAR>                                      CACharRWQueue;


/// UNICODE ַд
typedef CRWQueue<WCHAR>                                     CWCharRWQueue;


#ifdef UNICODE

/// ַд
typedef CWCharRWQueue                                       CCharRWQueue;

#else

/// ַд
typedef CACharRWQueue                                       CCharRWQueue;

#endif


/////////////////////////////////////////////////////////////////////////////


#endif